/*
 * Copyright (C) 2013 Google Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "CSSToLengthConversionData.h"

#include "FloatSize.h"
#include "RenderStyle+GettersInlines.h"
#include "RenderView.h"
#include "StyleBuilderState.h"

namespace WebCore {

CSSToLengthConversionData::CSSToLengthConversionData() = default;
CSSToLengthConversionData::CSSToLengthConversionData(const CSSToLengthConversionData&) = default;
CSSToLengthConversionData::CSSToLengthConversionData(CSSToLengthConversionData&&) = default;

// FIXME: Only rely on the RenderView for style resolution if we have an active LocalFrameView.
static RenderView* renderViewForDocument(const Document& document)
{
    if (document.view()) [[likely]]
        return document.renderView();
    return nullptr;
}

CSSToLengthConversionData::CSSToLengthConversionData(const RenderStyle& style, Style::BuilderState& builderState)
    : m_style(&style)
    , m_rootStyle(builderState.rootElementRenderStyle())
    , m_parentStyle(&builderState.parentRenderStyle())
    , m_renderView(renderViewForDocument(builderState.document()))
    , m_elementForContainerUnitResolution(builderState.element())
    , m_styleBuilderState(&builderState)
{
}

CSSToLengthConversionData::CSSToLengthConversionData(const RenderStyle& style, const RenderStyle* rootStyle, const RenderStyle* parentStyle, const RenderView* renderView, const Element* elementForContainerUnitResolution)
    : m_style(&style)
    , m_rootStyle(rootStyle)
    , m_parentStyle(parentStyle)
    , m_renderView(renderView)
    , m_elementForContainerUnitResolution(elementForContainerUnitResolution)
    , m_zoom(1.f)
{
}

CSSToLengthConversionData::~CSSToLengthConversionData() = default;

const FontCascade& CSSToLengthConversionData::fontCascadeForFontUnits() const
{
    if (computingFontSize()) {
        ASSERT(parentStyle());
        return parentStyle()->fontCascade();
    }
    ASSERT(style());
    return style()->fontCascade();
}

float CSSToLengthConversionData::computedLineHeightForFontUnits() const
{
    if (computingFontSize()) {
        ASSERT(parentStyle());
        return parentStyle()->computedLineHeight();
    }
    ASSERT(style());
    return style()->computedLineHeight();
}

float CSSToLengthConversionData::zoom() const
{
    return m_zoom.value_or(m_style ? m_style->usedZoom() : 1.f);
}

FloatSize CSSToLengthConversionData::defaultViewportFactor() const
{
    if (m_styleBuilderState)
        m_styleBuilderState->setUsesViewportUnits();

    if (!m_renderView)
        return { };

    return m_renderView->sizeForCSSDefaultViewportUnits() / 100.0;
}

FloatSize CSSToLengthConversionData::smallViewportFactor() const
{
    if (m_styleBuilderState)
        m_styleBuilderState->setUsesViewportUnits();

    if (!m_renderView)
        return { };

    return m_renderView->sizeForCSSSmallViewportUnits() / 100.0;
}

FloatSize CSSToLengthConversionData::largeViewportFactor() const
{
    if (m_styleBuilderState)
        m_styleBuilderState->setUsesViewportUnits();

    if (!m_renderView)
        return { };

    return m_renderView->sizeForCSSLargeViewportUnits() / 100.0;
}

FloatSize CSSToLengthConversionData::dynamicViewportFactor() const
{
    if (m_styleBuilderState)
        m_styleBuilderState->setUsesViewportUnits();

    if (!m_renderView)
        return { };

    return m_renderView->sizeForCSSDynamicViewportUnits() / 100.0;
}

void CSSToLengthConversionData::setUsesContainerUnits() const
{
    if (m_styleBuilderState)
        m_styleBuilderState->setUsesContainerUnits();
}

CheckedPtr<Style::BuilderState> CSSToLengthConversionData::protectedStyleBuilderState() const
{
    return m_styleBuilderState;
}

bool CSSToLengthConversionData::evaluationTimeZoomEnabled() const
{
    ASSERT(m_style);
    return CheckedPtr { m_style }->evaluationTimeZoomEnabled();
}

} // namespace WebCore
