const std = @import("std");

// Although this function looks imperative, note that its job is to
// declaratively construct a build graph that will be executed by an external
// runner.
pub fn build(b: *std.Build) void {
    // Standard target options allows the person running `zig build` to choose
    // what target to build for. Here we do not override the defaults, which
    // means any target is allowed, and the default is native. Other options
    // for restricting supported target set are available.
    const target = b.standardTargetOptions(.{});

    // Standard optimization options allow the person running `zig build` to select
    // between Debug, ReleaseSafe, ReleaseFast, and ReleaseSmall. Here we do not
    // set a preferred release mode, allowing the user to decide how to optimize.
    const optimize = b.standardOptimizeOption(.{});

    const pie = b.option(bool, "pie", "Build a Position Independent Executable") orelse true;
    const relro = b.option(bool, "relro", "Force all relocations to be read-only after processing") orelse true;

    // load the "zig-toml" dependency from build.zig.zon
    const toml_package = b.dependency("zig_toml", .{
        .target = target,
        .optimize = optimize,
    });
    const toml_module = toml_package.module("toml");

    const bearssl_package = b.dependency("zig_bearssl", .{
        .target = target,
        .optimize = optimize,
    });
    const bearssl_module = bearssl_package.module("bearssl");

    //const lib = b.addStaticLibrary(.{
    //    .name = "server",
    //    // In this case the main source file is merely a path, however, in more
    //    // complicated build scripts, this could be a generated file.
    //    .root_source_file = b.path("src/root.zig"),
    //    .target = target,
    //    .optimize = optimize,
    //});

    //// This declares intent for the library to be installed into the standard
    //// location when the user invokes the "install" step (the default step when
    //// running `zig build`).
    //b.installArtifact(lib);

    const exe = b.addExecutable(.{
        .name = "klutshnikd",
        .root_source_file = b.path("src/main.zig"),
        .target = target,
        .optimize = optimize,
        .linkage = .static,
    });

    exe.pie = pie;
    exe.link_z_relro = relro;
    exe.bundle_compiler_rt = true;
    exe.root_module.addImport("toml", toml_module);
    exe.root_module.addImport("bearssl", bearssl_module);
    exe.linkLibrary(bearssl_package.artifact("zig-bearssl"));
    //exe.linkSystemLibrary2("oprf-noiseXK", .{ .preferred_link_mode = .static });
    //exe.linkSystemLibrary2("oprf", .{ .preferred_link_mode = .static });
    //exe.linkSystemLibrary2("sodium", .{ .preferred_link_mode = .static });
    //exe.linkSystemLibrary("oprf");
    //exe.linkSystemLibrary("sodium");
    exe.addObjectFile(.{ .cwd_relative = ("/usr/lib/libsodium.a") });
    exe.addObjectFile(.{ .cwd_relative = ("/usr/lib/liboprf.a") });
    exe.addObjectFile(.{ .cwd_relative = ("/usr/lib/liboprf-noiseXK.a") });
    exe.addSystemIncludePath(.{ .cwd_relative = "/usr/include/oprf/noiseXK/" });
    exe.addSystemIncludePath(.{ .cwd_relative = "/usr/include/oprf/noiseXK/karmel" });
    exe.addSystemIncludePath(.{ .cwd_relative = "/usr/include/oprf/noiseXK/karmel/minimal" });
    exe.addIncludePath(b.path("."));
    exe.addCSourceFile(.{ .file = b.path("src/workaround.c"), .flags = &[_][]const u8{"-Wall"} });
    exe.addCSourceFile(.{ .file = b.path("cc-runtime/cc-runtime.c"), .flags = &[_][]const u8{"-Wall"} });
    //exe.linkLibC();

    // This declares intent for the executable to be installed into the
    // standard location when the user invokes the "install" step (the default
    // step when running `zig build`).
    b.installArtifact(exe);

    // This *creates* a Run step in the build graph, to be executed when another
    // step is evaluated that depends on it. The next line below will establish
    // such a dependency.
    const run_cmd = b.addRunArtifact(exe);

    // By making the run step depend on the install step, it will be run from the
    // installation directory rather than directly from within the cache directory.
    // This is not necessary, however, if the application depends on other installed
    // files, this ensures they will be present and in the expected location.
    run_cmd.step.dependOn(b.getInstallStep());

    // This allows the user to pass arguments to the application in the build
    // command itself, like this: `zig build run -- arg1 arg2 etc`
    if (b.args) |args| {
        run_cmd.addArgs(args);
    }

    // This creates a build step. It will be visible in the `zig build --help` menu,
    // and can be selected like this: `zig build run`
    // This will evaluate the `run` step rather than the default, which is "install".
    const run_step = b.step("run", "Run the app");
    run_step.dependOn(&run_cmd.step);

    // Creates a step for unit testing. This only builds the test executable
    // but does not run it.
    //const lib_unit_tests = b.addTest(.{
    //    .root_source_file = b.path("src/root.zig"),
    //    .target = target,
    //    .optimize = optimize,
    //});

    //const run_lib_unit_tests = b.addRunArtifact(lib_unit_tests);

    //const exe_unit_tests = b.addTest(.{
    //    .root_source_file = b.path("src/main.zig"),
    //    .target = target,
    //    .optimize = optimize,
    //});

    //const run_exe_unit_tests = b.addRunArtifact(exe_unit_tests);

    // Similar to creating the run step earlier, this exposes a `test` step to
    // the `zig build --help` menu, providing a way for the user to request
    // running the unit tests.
    //const test_step = b.step("test", "Run unit tests");
    //test_step.dependOn(&run_lib_unit_tests.step);
    //test_step.dependOn(&run_exe_unit_tests.step);
}
