%feature("docstring") OT::IterativeExtrema
"Iterative minimum and maximum.

Parameters
----------
dimension : int
    Dimension of the input data

Notes
-----
This class iteratively computes the minimum and maximum over an iteratively increasing
dataset without storing any data in memory.

See also
--------
IterativeAlgorithm

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> dim = 5
>>> myExtrema = ot.IterativeExtrema(dim)
>>> n = ot.Normal(dim)
>>> size = 50

Increment the data one point at a time:

>>> for i in range(size):
...     point = n.getRealization()
...     myExtrema.increment(point)
>>> print(myExtrema.getIterationNumber())
50
>>> print(myExtrema.getMin())
[-2.4067,-2.53986,-2.29006,-3.09737,-2.18139]
>>> print(myExtrema.getMax())
[3.01263,3.02799,1.85579,2.11968,1.36783]

Increment with 50 additional points simultaneously and recompute the extrema:

>>> sample = n.getSample(size)
>>> myExtrema.increment(sample)
>>> print(myExtrema.getIterationNumber())
100
>>> print(myExtrema.getMin())
[-2.4067,-2.53986,-2.72106,-3.09737,-2.18139]
>>> print(myExtrema.getMax())
[3.01263,3.02799,2.24097,2.11968,2.55533]"

// ---------------------------------------------------------------------

%feature("docstring") OT::IterativeExtrema::getMin
"Returns the values of the minimum over the inputs component per component

Returns
-------
min : :class:`~openturns.Point`
    current values of the iterative minimum."

// ---------------------------------------------------------------------

%feature("docstring") OT::IterativeExtrema::getMax
"Returns the values of the maximum over the inputs component per component

Returns
-------
max : :class:`~openturns.Point`
    current values of the iterative maximum."
