# USAGE:  gawk -f "this-file" Shelp.nroff
#          
# translates [nt]roff S/S-Plus documentation to SGML according to the 
# DTD in $SHOME/library/documentation/dtd/s-function-doc.dtd
# For details, see the Notes at the end of the file

BEGIN	{ initialize() }

/^[\.']\\["']/ { next  }				# skip troff comments
/^[^\.]/ { processLine(state, $0); next }		# simple text
#
# Support help hyperlinks in main text
#    Usage:  .HelpLink foo				# Help file link
#
/^\.HelpLink/ { HyperLink(stripQuotes($2), "function"); printf("\n"); next }
/^\.Helplink/ { HyperLink(stripQuotes($2), "function"); printf("\n"); next }
#
#   Additions by Tim H, to support hyperlinks and verbatim sgml
#
/^\.beginSGML/ { beginSGML();   next }			# Verbatim sgml start
/^\.endSGML/   { endSGML();     next }			# Verbatim sgml end
/^\.beginHTML/ { beginHTML();   next }			# Verbatim html start
/^\.endHTML/   { endHTML();     next }			# Verbatim html end
#
#   Additions by Tim H, for avoiding some section titles
#
/^\.SAo/ { initState(".SAo"); next }			# See Also mode, no hdr
/^\.SHo/ { closeState(state); newParagraph(); next }	# section, no hdr
#
#   Extract the major sections 
#
/^\.BG/ { beginDoc();       next }			# Begin documentation
/^\.FN/ { FunNames[++Nfuns]=$2;  next }		 	# Function name (topic)
/^\.TL/	{ initState(".TL"); next }			# Title 
/^\.DN/ { initState(".DN"); next }			# Description
/^\.CS/	{ initState(".CS"); next }			# Calling Sequence
/^\.CM/ { initState(".CS"); next }			# Splus uses this as CS
#
#   Argument Processing.  This can be tricky, especially since a lot of
#   existing help files incorrectly use argument macros in other sections.
#
/^\.Ag/ { initState(".Ag"); next }			# Argument section
/^\.OA/	{ initState(".OA"); next }			# Optional Arguments 
/^\.RA/	{ initState(".RA"); next }			# Requiered Arguments
/^\.AG/ { printArg(stripQuotes(stripTag($0))); next }	# Argument name
/^\.IP/	{ printArg(stripQuotes(stripClean($2))); next }
/^\.IB/	{ beginItemize();    next }			# Splus Itemize Begin
/^\.IT/ { itemMark($2);      next }			# Splus Item
/^\.IE/ { endItemize();      next }			# Splus Itemize End
#
#   no more argument processing, back to major sections
#
/^\.DT/	{ initState(".DT"); next }			# Details
/^\.SE/	{ initState(".SE"); next } 			# Side Effects
/^\.RT/	{ initState(".RT"); next } 			# Returned value
/^\.RC/ { printRC(stripQuotes(stripTag($0)));  next }	# Return Component
/^\.EX/	{ initState(".EX"); next } 			# Example 
/^\.SA/ { initState(".SA"); next }			# See Also
/^\.KW/ { processLine(".KW", $2); next }		# Key Word 
/^\.WR/	{ endDoc();         next }			# Wrapup
#
#   Done with major sections. The following are miscellaneous
#   optional sections.
#
/^\.SH/	{ tag = SH2sec($0); initState(tag); next }	# Section Heading
/^\.CA/ { printMsg(".CA", $2); next }			# Class and group
/^\.Ca/ { printMsg(".Ca");     next }			# Class
#
#   The following are close in spirit to SGML's general entities
#   (mostly involve simple text replacement).
#
/^\.GR/	{ printMsg(".GR"); next }			# Graphical Parameters 
/^\.SP/	{ printMsg(".SP"); next }			# Single Precision
/^\.Tl/	{ printMsg(".Tl"); next }			# args to title
/^\.NA/	{ printMsg(".NA"); next }			# NA's allowed
/^\.GE/	{ printMsg(".GE"); next }			# Generic function
/^\.ME/	{ printMsg(".ME", $2, $3); next }		# Method generic/group
/^\.AO/ { printMsg(".AO",$2); next }	# Arg to Other fun
/^\.XY/	{ printArg("x,y"); printMsg(".XY");    next }  	# XY coords
/^\.PA/ { processLine(".Ag", "...")			# Splus Printer Arg
	  printMsg(".PA")
          next 
 	}
#
#   The following can appear inside any of the high-level elements (sections).
#
/^\.PP/	{ newParagraph(); next }			# Paragraph
/^\.br/	{ newParagraph1(); next }			# Paragraph
/^\.sp/	{ newParagraph(); next }			# Paragraph
/^\.Cs/	{  beginCode();    next }			# Code Start
/^\.Ce/	{  endCode();     next }			# Code end
/^\.I /	{ printf("%s", inFont(".I",  $0)); next }	# font Italics
/^\.B /	{ printf("%s", inFont(".B",  $0)); next }	# font Bold
/^\.Sc/	{ printf("%s", inFont(".Sc", $0)); next }	# font Small Caps
/^\.Co/ { printf("%s", inFont(".Co", $0)); next }	# font Computer output
/^\.Cq/ { printf("%s", inFont(".Cq", $0)); next }	# font Computer quote
/^\.ul/	{ getline;     underLine($0);      next }	# underline
/^\.(sp|br|ti)/	{ lineBreak(); next }			# these generate breaks
/^\.ce/	{ n=($NF==1)?1:$2; beginCenter(n); next}	# center n lines
/^\.Si/	{ printSinput( stripTag($0) ); next }		# S input 
#
#   troff-specific output 
#
/^\.Tr/	{ beginTroff(); next }				# troff-specific code  
/^\.En/ { endTroff();   next }				# end of troff-specific
/^\.Tf/ { beginTroff(); next }
/^\.F\+/{ beginTroff(); next }
/^\.F\-/{ endTroff();   next }
/^\.Hl/ { beginNroff(); next }  			# only for help() fun
/^\.nf/	{ beginNofill(); next }				# no-fill 
/^\.fi/ { endNofill();   next }				# fill    
/^\.EQ/ { beginTroff(); next }				# EQN start
/^\.EN/ { endTroff();   next }				# EQN end
/^\.Nx/ { processLine(state, $0); next }
/^\.VE/	{ processLine(state, $0); next }		# Version? (deprecated)
/^\.PI/ { processLine(state, $0); next }		# Picture Inclusion
/^\.ne/	{ processLine(state, $0); next }		# hope is in troff mode!
/^\.Xp/ { processLine(state, $0); next }		# Unknown 
/^\./   { processLine(state, $0); next }		# any other troff cmd
        { if(state==".EX" || state==".CS")
	  processLine(state, $0); next }
# 
# No more parsing.  The rest of the file defines output functions.
#
function beginDoc(){ 	# it's posible to have troff before .BG (Grrrr!!)
   if(inDoc==0){
     printf("<!doctype s-function-doc system \"s-function-doc.dtd\" [\n<!entity %% S-OLD \"INCLUDE\">\n]\n>\n")
     printf("<s-function-doc>\n")
   }
   inDoc = 1
}
function endDoc(){  
   closeState(state)
   printf("<s-docclass>\nfunction\n</s-docclass>\n")
   printf("</s-function-doc>\n")
}

# define various "environments" (can occur *inside* sections ("states"))
function beginCode()    { inCodeEnv = 1; printf("<verb>\n"); }
function endCode()      { inCodeEnv = 0; printf("</verb>\n");  }

# Next four added by Tim H, for verbatim sgml and html
function beginSGML()    { inCodeEnv = 1; newParagraph(); }
function endSGML()      { inCodeEnv = 0; newParagraph();  }
function beginHTML()    { inCodeEnv = 1; newParagraph();printf("<![ CDATA [ ");}
function endHTML()      { inCodeEnv = 0; printf("]]>\n"); newParagraph();}

function beginNofill() 	{ inNofillEnv = 1; newParagraph(); }
function endNofill()   	{ inNofillEnv = 0; newParagraph(); }
function beginNroff(){ #this too can occur outside .BG/.WR
  if(inDoc==0) beginDoc()
    inNroffEnv++ # allow for nesting on input and on output
    if(inNroffEnv==1) printf("<!--Alternate nroff text-->\n")
}
function beginTroff(){	# this can even occur outside .BG/.WR
  if(inDoc==0) beginDoc()
    inTroffEnv++	# allow for nesting on input (AND on output)
    if(inTroffEnv==1) printf("<!-- <special target=\"troff\">\n%s\n", $0)
}
function endTroff(){ 
      if (inNroffEnv>=1)  
	  inNroffEnv--
      if (inNroffEnv==0 && inTroffEnv==0){
	printf("<!--End alternate nroff text-->\n")
	return
	}
  if(inTroffEnv==0)
    return
  inTroffEnv--
  if(inTroffEnv==0) printf("%s\n</special> \n -->\n",$0)
}

# not used by SGML (but handy in HTML)
function beginIgnore()  { inIgnoreEnv=1 }
function endIgnore()    { inIgnoreEnv=0 }
function beginCenter(n)	{ inCenterEnv=1; printf("<!--center-->\n"); }
function endCenter()  	{ inCenterEnv=0; printf("<!--/center-->\n");}

function beginArgEnv()  { inArgEnv=1  }   
function endArgEnv()    { inArgEnv=0; printf("%s\n", clTag[".AG"])  }

function beginRCEnv()	{ inRCEnv=1  }   
function endRCEnv()	{ inRCEnv=0; printf("%s\n", clTag[".RC"])  }

# These generate output according to linuxdoc's DTD
function beginItemize() { inItemizeEnv=1; printf("<descrip>\n")    }
function endItemize()   { inItemizeEnv=0; printf("</descrip>\n")   }
function itemMark(m){ 
  sub(/\\\(bu/, "\\&bull;", m)			# bullet symbol
  m = filter(m)
  printf("<tag>%s</tag>\n", m)
}
function printArg(a){ 
  if(state==".RT"){	# Grr! deal w. badly written help 
	printRC(a)
	return
  }
  if(state!=".Ag" && state!=".OA" && state!=".RA")
  	initState(".Ag")  		# init Arg state 
  arg = filter(a)
  gsub(/"/, "'", arg)			# double quotes cannot appear here
  if(inArgEnv==1)
    printf("%s\n", clTag[".AG"])	# close previous arg tag
  printf(opTag[".AG"] "\n", arg)	# open new arg tag
  inArgEnv = 1
}
function printRC(rc){			# mimic printArg (could combine them)
  # The next line causes problms in .SH STRUCTURE
  #  initState(".RT")
  # Instead, add this line, for a <DL> tag pair
  initState(".RTo")
  rc = filter(rc)
  if(inRCEnv==1)
    printf("%s\n", clTag[".RC"])	# close previous RC tag
  printf(opTag[".RC"] "\n", rc)		# open RC tag 
  inRCEnv = 1
}
function printSinput(e){ 
  printf("&gt; <code>%s</code>\n", stripTag(e))
}
function lineBreak()	 { printf("\n");  }
function underLine(item) { printf("<it>%s</it>\n", replSgmlEnt(item)) }
function newParagraph()  {
   if(state==".CS" || state==".EX") {
       printf("\n")
       if(state==".CS")
       closeState(state)
      return
  }
      printf("<p>\n")#  printf("\n");
}
function newParagraph1()  {
   # like newParagraph, but only one <br>
   if(state==".CS" || state==".EX") {
       printf("\n")
       if(state==".CS")
       closeState(state)
      return
  }
      printf("<br>\n")#  printf("\n");
}
function stripTag(s){		# remove troff tag (command) from s
  sub(/^\.[A-Z][A-Za-z]? */, "", s)  
  return s 
}
function stripSH(s){        # remove SH from section header
   gsub(".SH", "", s)	
   return s
}
function stripQuotes(s) {	# remove back, single, and double quotes
  gsub(/(^[`'"]|[`'"]|[']$)/,"", s)
  return s
}
function stripDot(s) {	        # remove initial .
  gsub(/^\./,"", s)
  return s
}
function stripPunct(s){		# remove punctuation 
  sub(/[,\.:;\?!]$/, "", s)
  return s
}
function stripClean(s){ 	# remove punctuation and quotes
  return stripQuotes(stripPunct(s)) 
}
function Debug(s){
  printf("<!--DEBUG: %s-->\n", s)
}
function Warning(s){
  printf("<!--WARNING: %s at input line %d-->\n", s, NR) | "cat 1>&2"
}
function initialize(	n, misc, tmp){
  # map the following .SH HEADING sections into SGML <s-heading> entities.
  misc = "BACKGROUND,BUGS,NOTE,NOTES,REFERENCES,WARNING,WARNINGS"
  n = split(misc, tmp, ",")
  for(i=1; i<=n; ++i)
    otherSecs[tmp[i]] = i
  Nfuns = 0;
  if(opTag[".Ag"]!="")	# in case of multiple files, only the above 
	return		# need re-initialization.
  #
  # Open/close tags for defining section-specific and environments (these 
  # could be easily read from mapping files)
  #
  opTag[".FN"] = "<s-topic>"         ; clTag[".FN"] = "</s-topic>"
  opTag[".TL"] = "<s-title>"         ; clTag[".TL"] = "</s-title>"
  opTag[".Ag"] = "<s-args>"          ; clTag[".Ag"] = "</s-args>"
  opTag[".RA"] = "<s-args-required>" ; clTag[".RA"] = "</s-args-required>"
  opTag[".OA"] = "<s-args-optional>" ; clTag[".OA"] = "</s-args-optional>"
  opTag[".CS"] = "<s-usage>\n<s-old-style-usage>"   
  clTag[".CS"] = "</s-old-style-usage>\n</s-usage>"
  opTag[".DN"] = "<s-description>"   ; clTag[".DN"] = "</s-description>"
  opTag[".DT"] = "<s-details>"       ; clTag[".DT"] = "</s-details>"
  opTag[".RT"] = "<s-value>"         ; clTag[".RT"] = "</s-value>"
  opTag[".SE"] = "<s-side-effects>"  ; clTag[".SE"] = "</s-side-effects>"
  opTag[".EX"] = "<s-examples>\n<s-example type = text>" 
  clTag[".EX"] = "</s-example>\n</s-examples>"
  opTag[".SA"] = "<s-see>"           ; clTag[".SA"] = "</s-see>"
  opTag[".KW"] = "<s-keywords>"      ; clTag[".KW"] = "</s-keywords>"
  opTag[".SH"] = "xxxxxxx"           ; clTag[".SH"] = "</s-section>"
  opTag[".AG"] = "<s-arg name=\"%s\">"	# this is used as a fmt in printf
  clTag[".AG"] = "</s-arg>"
  opTag[".RC"] = "<s-return-component name=\"%s\">"
  clTag[".RC"] = "</s-return-component>"
  opTag[".SAo"] = "\n"                  ; clTag[".SAo"]= "\n"
  # opTag[".RTo"] = "<descrip>"         ; clTag[".RTo"] = "</descrip>"
  # That fails.  Need <DL> and </DL> in the html; next two lines are ugly,
  # supply the html directly:
  opTag[".RTo"] = "<![ CDATA [ <DL>   ]]>\n"
  clTag[".RTo"] = "<![ CDATA [ </DL>  ]]>\n"

  #
  # Originally, available fonts for help() depended on whether output was 
  # to be processed by nroff or troff (on-line or off-line).  
  # SGML/HTML really don't care about this.
  #
  troffBindings = 0;			# output suitable for nroff
  if(troffBindings){
    nFonts = split("R,I,C,B,C,C,C,C,C", inFonts, ",")	# input help.tr
    nFonts = split("r,it,tt,bf,tt,tt,tt,tt,tt", outFonts, ",") 
  } else {
    nFonts = split("R,I,B,S,C",     inFonts, ",")		# input help.nr
    nFonts = split("r,it,bf,bf,tt", outFonts, ",")		# output linuxdoc fonts
  }
  currFont = 1; prevFont = 0;
  for(i=1; i<= nFonts; i++){
    fontPosition[inFonts[i]] = i
    opFont[i] = sprintf("<%s>",  outFonts[i])	# e.g., <tt></tt>
    clFont[i] = sprintf("</%s>", outFonts[i])
  }
  TitleDone = 0
}
function inFont(font, item){
  # sandwich "item" between SGML font tags, e.g., <tt>item</tt>
  item = stripQuotes(stripTag(item))	# remove troff cmd and quotes
  if(font==".I") f = "I"
  else if(font==".Co"||font==".Cq") f = "C"
  else if(font==".B"||font==".Sc") f = "B"
  else f = "R"
  i = fontPosition[f]
  return sprintf("<%s>%s</%s>", outFonts[i], replSgmlEnt(item), outFonts[i])
}
#
# create a reference to an s-object (perhaps this function should 
# return a string?)
#
function HyperLink(name, type,		punct){
  # "type" is one of "function", "class", "object", "method"
  # drop quotes (single and double) and punctuation in the link itself,
  # but put punctuation back after the link.
  if( match(name, /[,.;:?!)]+$/) ){		# any punctuation?
	punct = substr(name, RSTART, RLENGTH)
	name  = substr(name, 1, RSTART-1)
  } else punct = ""
  name = stripQuotes(name)
  printf("<s-%s name=\"%s.sgm\">%s</s-%s>%s ",type, stripDot(name), name, type,punct)
}
function initState(s){
  # init a new state **if needed only***
  if(s==".FN") {
	FunNames[++Nfuns] = $2		# fun names (i.e., topics)
        if(TitleDone)                  # out-of-place .Fn; emit separately
	      printf("<s-topics>\n<s-topic>%s</s-topic>\n</s-topics>\n", $2)
  }
  if(state==s)
	return				# we're in the state already
  if(inTroffEnv==1){
	processLine(s, $0)
	return
  }
  if(state!=s)
	closeState(state)	 	# here we also end environments
  if(s==".TL" && Nfuns > 0){		# flush <s-topics> before title line
        printf("<s-topics>\n")
	for(i=1; i<=Nfuns; i++)
	  printf("   %s%s%s\n", opTag[".FN"], FunNames[i], clTag[".FN"])
        printf("</s-topics>\n")
  }
  if(opTag[s]!="" && s!=".SH") 
	printf("%s\n", opTag[s])
  if((s==".SA"||s==".SAo"||s==".GE") && NF > 0){
	for(i=2; i<=NF; i++) HyperLink($i, "function")
  }
  state = s
  return
}
function closeState(s){
  if(inCodeEnv>0)    endCode()
  if(inNofillEnv>0)    endNofill()
  if(inTroffEnv>0)   endTroff()
  if(inNroffEnv>0)   endTroff()
  if(inCenterEnv>0)  endCenter()
  if(inItemizeEnv>0) endItemize()
  if(inArgEnv>0)     endArgEnv()
  if(inRCEnv>0)      endRCEnv()
  if(clTag[s]!="")   printf("%s\n", clTag[s])
  resetFont()
  if(s==".TL") ## mark this so any .FN's after will be output immediately
	TitleDone = 1
  state = ""
  return
}
function printHeading(t) { 
   SH2sec(t) 
}
#
# map some .SH FOO sections into their own SGML tags <s-foo></s-foo>
# (see the array "otherSecs" for the complete list)
#
function SH2sec(t,    sec, tmp0, tmp, n, flds){
  tmp0 = stripQuotes(t)
  tmp = toupper(tmp0)
  n = split(tmp, flds)
  sec = (n>1) ? flds[1] : ".SH"
  if(sec in otherSecs)
    opTag[sec] = sprintf("<s-%s>",  tolower(sec))
  else  {
    tmp = stripQuotes(stripSH(tmp0))
    gsub(/"/,"",tmp)
    opTag[sec] = sprintf("<s-section name=\"%s\">\n", tmp)
    closeState(state)
    printf("%s\n", opTag[sec])	
  }
  initState(sec)
  return sec
}
function replSgmlEnt(s){
  gsub("&", "\\&#38;", s)		# for HTML and SGML's sake...
  gsub("<", "\\&lt;",  s)
  gsub(">", "\\&gt;",  s)
  return s
}
function processLine(st, line){
  # output line in accordance to state/section "st", also checks whether
  # we're inside an environment 
  if(inDoc!=1) beginDoc()		# it's possible to have troff before .BG
  if(inIgnoreEnv==1) return		# bliss!

  if(inTroffEnv==1){
  	printf("%s\n", $0)		# copy raw troff
	return
  }
  if(st!="") initState(st) 		# initialize if needed

  if(st==".SA"||st==".SAo"||st==".GE") {	# make input into links to funs
    if(NF==1&&st==".GE"){
        printf("\n")
	HyperLink($1, "function")
    }
    else {
	for(i=1; i<=NF; i++) {
  	  if(match($i, /`[A-Za-z\.0-9\$].*'/)){
	    printf("\n")
	    HyperLink($i, "function")
	    printf(" ")
	  }
	  else printf("%s ", filter($i))
        }
    }
  } else if(st==".KW"){
	  printf("<s-keyword>%s</s-keyword>\n", filter(line))
    } else if (st == ".DN"){
       l = filter(line)
       gsub(/"/, "", l)
       printf("%s \n",l)
    } else if(st!=".FN"){		# .FN is saved for later output
          printf("%s \n", filter(line))
    } else if(st==".EX" || st==".CS"){
	  printf("%s \n",filter(line))
  }
  return
}
function filter(line,		font, n, tmp) {
  # filter input for some HTML entities, unless we are in code/verbatim;
  # also deal with the tricky troff font changing problem
  if(inTroffEnv) 
	return line
  gsub(/\\e/,"\\",line)
  gsub(/\\\(em/,"---",line)               
  gsub(/\\ /," ",line)			# slash space
  gsub(/\\\&/, "", line)		# annoying troff \&
  gsub(/\\\^/, "", line)		# \^ is a small space (1/6 em)
  gsub(/\\\|/, "", line)		# \| is a tiny space (1/12 em)
  gsub(/\\-/, "--",line)		# minus char
  gsub(/\\0/, " ", line)		# one-digit width--eh?
  if(inCodeEnv==1){
	# remove font changes (why would anybody change fonts here? Yet...)
	gsub(/\\f[0-8PRICB]/, "", line)
	return line
  }
  if(state != ".EX")
    line = replSgmlEnt(line)		# replace <, >, &
  # replace `abc' by <code>abc</code> (or <s-function>)
  if(state==".SA"||state==".SAo")
	type = "function"
  else  {
    if (state == ".EX" || state == ".CS")
	    return line
    else  type = "expression"
  }


  if( type == "function")
  str  = sprintf("<s-%s name = &.sgm>&</s-%s>", type, type)      
  else
  str  = sprintf("<code>&</code>")      
  # In previous line, assume type=expression; use code instead of s-expression
  # if(match(line, /[`'][A-Za-z0-9\_\.\:\~\@\!\%\&\|\;\#\{\}\=\(\)\[\$\^\+\*\/\",\ \-]+[']/)){
  # awk on OSF/1 and AIX doesn't recognize ] in pattern [\[\]]
  if(match(line, /[`'][^`']+[']/)){
  	    gsub(/[`'][^`']+[']/, str, line)
  	    # gsub(/[`'][A-Za-z0-9\_\.\:\~\@\!\%\&\|\;\#\{\}\=\(\)\[\$\^\+\*\/\",\ \-]+[']/, str, line)
	    gsub(/[`']/, "", line)
  }
  # Grrrr! need to identify font changes:-(
  while(n=match(line, /\\f[0-9PRICB]/)){	# BUG: misses \f(XX
    # replace \fy by <y>, possibly </x><y>
    if(isFontOpen>0){
    	tmp = clFont[currFont] 
	--isFontOpen
    } else tmp=""
    font = substr(line,n+2,1)
    if(font=="0" || font=="P"){
	font = prevFont
    } else { 
	if(font~/1-9/)
	   font = int(font)
 	else font = fontPosition[font]
	if(font>1)
	   ++isFontOpen
    }
    if(font>1)		# SGML: linuxdoc doesn't have a <r> font as the default
    	tmp = tmp opFont[font]
    line= sprintf("%s%s%s",substr(line,1,RSTART-1),tmp,
		   substr(line, RSTART+RLENGTH))
    prevFont = currFont
    currFont = font
    }
  return line 
}
function resetFont(){ 
  if(isFontOpen>0)
	printf("%s\n", clFont[currFont])
  currFont = prevFont = 1
  isFontOpen = 0
}
function printMsg(tag, str1, str2){ 
  # print appropriate message for "tag", possibly with extra str1 and str2
  if(tag==".AO"){	  
     printf("<p>Arguments for function ")
     HyperLink(str1, "function")
     printf(" can also be supplied to this function.\n")
     return
  }
  if(tag==".XY"){
     printf("\ncoordinates of points. ")
     printf("The coordinates can be given by two vector " )
     printf("arguments or by a single arguments <s-object>x</s-object> ")
     printf("which is a times series, a two-column matrix, ")
     printf("or a list containing components named ")
     printf("<s-object>x</s-object> and <s-object>y</s-object>.\n")
     return
  }
  if(tag==".Tl"){
     printf("In addition, the high-level graphics arguments ")
     printf("described under ")
     HyperLink("plot.default", "function")
     printf(" and the arguments to ")
     HyperLink("title", "function")
     printf(" may be supplied to this function.\n")
     return
  }
  if(tag==".NA"){
     printf("\nMissing values <s-object>(NA)</s-object> are allowed.\n") 
     return
  }
  if(tag==".SP"){
     printf("\nResults are currently computed to ")
     printf("single-precision accuracy only.\n")
     return
  }
  if(tag==".GR"){ 
     printf("Graphical parameters may also be supplied as arguments to \n")
     printf("this  function (see ")
     HyperLink("par", "function")
     printf(").\n")
     return
  }
  if(tag==".GE"){
    printf("<p>This function is an S Version 3 generic (see Methods);\n ")
    printf("method  functions can  be  written  to  handle  specific \n")
    printf("S Version 3 classes  of  data. Classes which  already \n")
    printf("have  methods  for  this  function include:<br> \n", FunNames[1])
    return
  }
  if(tag==".CA"){
     printHeading("CLASSES")
     printf("This function will be used as the default method ")
     printf("for classes that do not inherit a specific method for the \n")
     printf("function or for the %s group of functions.\n",
	     inFont(".Co",str1) )
     printf("The result will retain the class and the attributes.\n")
     printf("If this behavior is <em>not</em> appropriate,\n")
     printf("the designer of the class should provide a method\n")
     printf("for the function or for the %s group\n", 
             inFont(".Co", str1))
     return
  }
  if(tag==".Ca"){
     printHeading("CLASSES")
     printf("This function will be used as the default method")
     printf("for classes that do not inherit a specific method for the\n")
     printf("function\n")
     printf("The result will retain the class and the attributes.\n")
     printf("If this behavior is <em>not</em> appropriate,\n")
     printf("the designer of the class should provide a method\n")
     printf("for the function.\n")
     return
  }
  if(tag==".ME"){
     printf("\n\nThis function is a method for the generic function ")
     HyperLink(str1, "function")
     printf("for class <s-class>%s</s-class>.\n", str2)
     printf("It can be invoked by calling")
     HyperLink(str1, "function")
     printf("for an object <s-object>x</s-object> of the appropriate class, ")
     printf("or directly by calling ")
     HyperLink( str1 "." str2 , "function")
     printf("regardless of the class of the object.\n")
     return
  }
  if(tag==".PA"){
     printf("other options are passed through to the printer drivers\n")
     printf("used by <s-function name= printgraph.sgm>printgraph</s-function>.\n")
     printf("Refer to <s-function name = ps.options>ps.options</s-function> for the list of\n")
     printf("valid commands for the PostScript driver.\n")
     printf("Each PostScript option should be prefixed\n")
     printf("with <tt>ps.</tt> in order to be passed through and\n")
     printf("recognized as a PostScript driver option.\n")
     printf("Note that the LaserJet driver has not yet been\n")
     printf("updated to use this scheme.\n")
     return
  }
}
 
# NOTES:  
# 1. If you want to apply this script to multiple files, make
#    sure that initialize() is called **for each file**, not in BEGIN.
# 2. We recognize 16 major Sections or SGML's elements 
# 3. There are 6 "environments" that can occur *anywhere*:
#    "code", "center", "itemize", "troff", "verbatim" and "ignore".
#    There are corresponding flags "inCodeEnv" ..., "inIgnoreEnv".
# 4. Most of the smarts (or whatever it has) is in the processLine()
#    and filter() functions, which have some heuristics to 
#    identify sctructured parts in the troff help files. 
# 5. For the troff macros .GR, .XY, .NA, .SP, ... that simply 
#    generate messages, we group them in the one fun printMsg().
# 6. Note that initState() is used on sections whose tags have no
#    parameters, e.g., .DT/.CS/.SA/.Ag/.OA/.RA.  Sections whose tags 
#    have params (.SH, .KW) need to be filtered by processLine(), which
#    it invokes initState().
# 7. It should be possible, with some effort, to identify troff contructs
#    used in help files that transalate into HTML tables (e.g., 
#    .nf		\" no-fill -- this is with together with the next line
#    .ta t1 t2 t3	\" tells us there are 3 cols (a tab signals the table)
#    col1   col2   col2 \" <td>col1<td>col2<td>col3
#    .br  		\" new row <tr>
#    .fi		\" </table>
#    The question is how to handle the exceptions to the above!
# 8. Known bugs: 
#	(1) two-letter font name changes are not recognized (\f(CBabc\fP)
#	(2) some troff comments are not preserved
#	(3) double quotes in arg names are replaced w. single quotes
#	(4) nested <code> are not caught
