/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

// splist.cxx: implementation of the CSPlist class.
//
//////////////////////////////////////////////////////////////////////
#include "S_y_tab.h"
#include "string.h"
#include "splist.h"
#include "spexcept.h"
#include "spchar.h"
#include "spalcfrm.h"


static const char* GenerateElementName(s_object* ps_existNames, const char* pszPrefix)
{
	if ( !pszPrefix || !*pszPrefix )
		return EMPTY_STRING;

	const char* pszName = pszPrefix;

	if ( SPL_NotThere(ps_existNames) )
		return pszPrefix;

	int iTry = 0;
	
	const char** arszNames = (const char**)CHARACTER_POINTER(ps_existNames);
	long n = GET_LENGTH(ps_existNames);
	do 
	{
		if ( iTry > 0 )
		{
			MESSAGE "%s%d", pszPrefix, iTry END_MESSAGE;
			pszName = GET_MESSAGE;
		}
		long i;
		for ( i=0; i<n; i++)
		{
			if ( strcmp(arszNames[i], pszName) == 0 )
				break;
		}
		if ( i >= n )
			return pszName;

		pszPrefix = pszName;

	} while ( ++iTry < 100 );

	return EMPTY_STRING;
}

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
CSPlist::CSPlist()
: TSPvector<s_object*, S_MODE_LIST>()
{
}
//Copy constructor 
CSPlist::CSPlist(const CSPlist& sObject)
: TSPvector<s_object*, S_MODE_LIST>()
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a base class object
CSPlist::CSPlist(const CSPobject& sObject)
: TSPvector<s_object*, S_MODE_LIST>()
{	
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a valid S-expression
CSPlist::CSPlist(const char* pszExpression)
: TSPvector<s_object*, S_MODE_LIST>(pszExpression)
{
}
//Construct from a valid S object
CSPlist::CSPlist(s_object* ps_object, BOOL bTryToFreeOnDetach)
: TSPvector<s_object*, S_MODE_LIST>()
{
	Attach(ps_object, bTryToFreeOnDetach);
}
//Assigment from the same class
CSPlist& CSPlist::operator=(const CSPlist& sObject)
{
	Attach(&sObject, TRUE);
	return *this;
}
//Assigment from the base class
CSPlist& CSPlist::operator=(const CSPobject& sObject)
{
	Attach(&sObject, TRUE);
	return *this;
}
//Assigment from the S object
CSPlist& CSPlist::operator=(s_object* ps_object)
{
	Attach(ps_object, FALSE);
	return *this;
}
//The destructor
CSPlist::~CSPlist()
{
}

//Construct a 'numeric' vector of length nLength
CSPlist::CSPlist(long lLength)
: TSPvector<s_object*, S_MODE_LIST>(lLength)
{
}

CSPlist::CSPlist(int nLength)
: TSPvector<s_object*, S_MODE_LIST>((long)nLength)
{
}

CSPlist::CSPlist(const CSPlistNamed& slistNamed)
: TSPvector<s_object*, S_MODE_LIST>()
{
	CSPevaluator sEvaluator;
	slistNamed.Validate(); //must be valid
	const CSPcharacter sNames(slistNamed.GetNames());
	long lLen = sNames.GetLength();
	s_object* ps_list = NewMyType(lLen);
	while(--lLen>=0)
	{
		LIST_POINTER(ps_list)[lLen] = sEvaluator.CreateNewHeader(slistNamed.GetAt(lLen));
		LIST_POINTER(ps_list)[lLen]->name = (char*)::make_name((char*)sNames[lLen], S_evaluator)->text;
	} 
	Attach(ps_list, TRUE);
}

//Assigment operator
CSPlist& CSPlist::operator= (const CSPlistNamed& slistNamed)
{
	CSPevaluator sEvaluator;
	slistNamed.Validate(); //must be valid
	const CSPcharacter sNames(slistNamed.GetNames());
	long lLen = sNames.GetLength();
	s_object* ps_list = NewMyType(lLen);
	while(--lLen>=0)
	{
		LIST_POINTER(ps_list)[lLen] = sEvaluator.CreateNewHeader(slistNamed.GetAt(lLen));
		LIST_POINTER(ps_list)[lLen]->name = (char*)::make_name((char*)sNames[lLen], S_evaluator)->text;
	} 
	ReAttachAndAssign(ps_list);
	return *this;
}

s_object*& CSPlist::GetAtDirect(long nZeroIndex, BOOL bValidate)
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	if ( nZeroIndex < 0 && nZeroIndex >= GetLength(bValidate) )
		SCONNECT_ThrowException("Invalid index %d", nZeroIndex);

	return GetPtr()->value.tree[nZeroIndex];
}

const s_object* CSPlist::GetAt(long nZeroIndex, BOOL bValidate) const
{
	if ( nZeroIndex < 0 && nZeroIndex >= GetLength(bValidate) )
		SCONNECT_ThrowException("Invalid index %d", nZeroIndex);

	return GetPtr()->value.tree[nZeroIndex];
}
s_object*& CSPlist::GetAt(long nZeroIndex, BOOL bValidate)
{
	if ( nZeroIndex < 0 && nZeroIndex >= GetLength(bValidate) )
		SCONNECT_ThrowException("Invalid index %d", nZeroIndex);

	return GetPtr()->value.tree[nZeroIndex];
}

void CSPlist::SetAt(long lZeroIndex, s_object* ps_element, BOOL bValidate)
{
	if(bValidate)
		Validate();
	BOOL bSuccess = FALSE;
	try
	{
		if(!ps_element)
			SCONNECT_ThrowException("NULL object");
		if(lZeroIndex < 0)
			SCONNECT_ThrowException("Invalid index %d", lZeroIndex);

		CSPevaluator sEvaluator;
		if(!IsSharing() && lZeroIndex < (*this)->nalloc )
		{
			//increase length without realloc
			if(lZeroIndex >= (*this)->length)
			{
				for(long i= (*this)->length; i <= lZeroIndex; ++i)
					(*this)->value.tree[i] = blt_in_NULL;
				(*this)->length = lZeroIndex+1L; 
			}
			sEvaluator.ReplaceElement(&(*this), lZeroIndex, ps_element);
			return;
		}
		s_object* ps_list = NULL;
		ps_list = CopyForWrite();
		if(IsSharing() && lZeroIndex < (*this)->nalloc )
			ps_list->value.tree[lZeroIndex] = sEvaluator.CreateNewHeader(ps_element);
		else  //Let append_data takes care of alloc mem.
			::append_el(ps_list, lZeroIndex, ps_element, S_evaluator);
		
		ps_list = sEvaluator.CloneIfNeeded(ps_list);		
		ReAttachAndAssign(ps_list);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
}
//TODO: perhaps use the same code as the engine to do name-indexing 
//and follow SetAt(long nZeroIndex,..) for protecting global object
void CSPlist::SetAt(const char* pszName, s_object* ps_element, BOOL bValidate)
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	BOOL bSuccess = FALSE;
	try
	{
		if ( !ps_element )
			SCONNECT_ThrowException("NULL object");
		if ( (pszName==NULL) || (*pszName == 0) )
			SCONNECT_ThrowException("Invalid name");

		//Find index for name, pszName
		s_object *ps_names = GetNames(FALSE);
		if ( SPL_NotThere(ps_names) )
			SCONNECT_ThrowException("List has no names");

		if ( !IS_CHARACTER(ps_names) )
			SCONNECT_ThrowException("List names are not character");

		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator s;

		long numNames = GET_LENGTH(ps_names);
		long nIndex;
		for ( nIndex=0; nIndex<numNames; nIndex++ )
		{
			if ( strcmp( CHARACTER_POINTER(ps_names)[nIndex], pszName ) == 0 )
				break;
		}
		s_object* ps_list = CopyForWrite();
		if ( GET_LENGTH(ps_list) <= nIndex )
			SET_LENGTH(ps_list, nIndex+1);

		if ( !IS_OBJ(ps_element,NULL) )
		{
			ps_element = ::s_header_for_object(ps_element, NULL, S_evaluator);
			if ( ps_element->name && strcmp(ps_element->name, pszName)!= 0 )
				ps_element->name = (char*)make_name(GenerateElementName(GetNames(FALSE),pszName),S_evaluator)->text;
		}
		LIST_POINTER(ps_list)[nIndex] = ps_element;
		ReAttachAndAssign(ps_list);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
}

void CSPlist::SetAtDirect(long lIndex, s_object* ps_element, BOOL bValidate)
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	if( !ps_element )
		SCONNECT_ThrowException("NULL object");
	if ( lIndex < 0 )
		SCONNECT_ThrowException("Invalid index %d", lIndex);
	if ( GET_LENGTH(GetPtr()) <= lIndex )
		SCONNECT_ThrowException("Invalid index %d: lIndex is greater than the object length %d", GET_LENGTH(GetPtr()));

	LIST_POINTER(GetPtr())[lIndex] = ps_element;

}

void CSPlist::SetAtDirect(const char* pszName, s_object *ps_element, BOOL bValidate)
{
	if ( !ps_element )
		SCONNECT_ThrowException("NULL object");
	if ( (pszName==NULL) || (*pszName == 0) )
		SCONNECT_ThrowException("Invalid name");

	//Find index for name, pszName
	s_object *ps_names = GetNames(bValidate);
	if ( SPL_NotThere(ps_names) )
		SCONNECT_ThrowException("List has no names");

	if ( !IS_CHARACTER(ps_names) )
		SCONNECT_ThrowException("List names are not character");

	long numNames = GET_LENGTH(ps_names);
	long lIndex;
	for ( lIndex=0; lIndex<numNames; lIndex++ )
	{
		if ( strcmp( CHARACTER_POINTER(ps_names)[lIndex], pszName ) == 0 )
			break;
	}

	if ( GET_LENGTH(GetPtr()) <= lIndex )
		SCONNECT_ThrowException("List has no names");

	LIST_POINTER(GetPtr())[lIndex] = ps_element;
}

BOOL CSPlist::InsertAt(s_object* ps_object, long lElement, const char* pszName, BOOL bValidate)
{
	BOOL bSuccess = FALSE;
	try
	{
		if(lElement<0)
			SCONNECT_ThrowException("Invalid element index", lElement, CSPexception::Error);
		if ( !ps_object )
			SCONNECT_ThrowException("Object to be inserted is invalid");

		CSPevaluator s;

		if(GetPtr() == NULL)
			SetLength(0); //Create an empty list

		long length = GetLength(bValidate);
		s_object* ps_list = CopyForWrite();

		long lenNew = (lElement>length?lElement+1:length+1);
		SET_LENGTH(ps_list, lenNew);

		for ( long i=1; i<=length-lElement; i++ )
		{
			LIST_POINTER(ps_list)[lenNew-i] = LIST_POINTER(ps_list)[length-i];
			LIST_POINTER(ps_list)[lenNew-i]->name = LIST_POINTER(ps_list)[length-i]->name;
			LIST_POINTER(ps_list)[length-i] = blt_in_NULL;
		}
		if ( !IS_OBJ(ps_object,NULL) )
		{
			char* pszName = ps_object->name;
			ps_object = SPL_NewHeaderForInsertion(ps_object);
			if ( pszName && *pszName )
				ps_object->name = (char*)make_name(GenerateElementName(GetNames(FALSE),pszName),S_evaluator)->text;
		}
		LIST_POINTER(ps_list)[lElement] = ps_object;

		s_object* ps_names = GetNames(FALSE);
		if ( pszName && *pszName )
			LIST_POINTER(ps_list)[lElement]->name = (char*)make_name(GenerateElementName(ps_names, pszName),S_evaluator)->text;
		else if ( ps_object->name && ps_object->name != EMPTY_STRING )
			LIST_POINTER(ps_list)[lElement]->name = (char*)make_name(GenerateElementName(ps_names, ps_object->name),S_evaluator)->text;

		ReAttachAndAssign(ps_list);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}

	return bSuccess;
}


long CSPlist::Add(s_object *ps_element, BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		Attach(alcvec(S_MODE_LIST,(long)0,S_evaluator));

	long nIndex = -1;
	try
	{
		if(!ps_element)
			SCONNECT_ThrowException("Adding a NULL element to a list");
		long n = GetLength(FALSE);
		if ( n < 0 )
			n = 0;
		SetAt( n, ps_element, FALSE );
		nIndex = n;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}

	return nIndex;
}


long CSPlist::Append( s_object* ps_list, BOOL bValidate )
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	long nIndex = -1;
	try
	{
		if( !IS_LIST(ps_list) )
			SCONNECT_ThrowException("Object to be appended must be a list");

		long nIndex = GetLength(FALSE);
		s_object* ps_data = GetPtr();
		if ( !ps_data )
			SCONNECT_ThrowException("List to be apended is missing a data slot");
		long appendLength = GET_LENGTH(ps_list);
		if ( appendLength <= 0 )
			return nIndex;
		for ( long i=0; i<appendLength; i++ )
		{
			if ( LIST_POINTER(ps_list)[i] != NULL )
			{
				long n = Add( LIST_POINTER(ps_list)[i], FALSE );
				if ( nIndex < 0 && n >= 0 )
					nIndex = n;
			}
		}
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}

	return nIndex;
}

BOOL CSPlist::SetNames(s_object *ps_names, BOOL bValidate)
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	BOOL bSuccess = FALSE;

	try
	{
		if ( !ps_names )
			SCONNECT_ThrowException("NULL object");
		BOOL bIsNULL = IS_OBJ(ps_names, NULL);
		if (  !bIsNULL )
		{
			if ( !IS_CHARACTER(ps_names) ) //Object of clas NULL is valid
				SCONNECT_ThrowException("Invalid names");
			if ( GET_LENGTH(ps_names) != GetLength(bValidate) )
				SCONNECT_ThrowException("Names are the wrong length", GET_LENGTH(ps_names));
		}

		CSPallocFrame sAllocFrame; //Set allocated frame to a perm frame if top-eval is not openned.

		s_object* ps_list = CopyForWrite();
		
		for ( long i=0; i<GET_LENGTH(ps_list); i++ )
		{
			s_object* ps_element = ::s_header_for_object(LIST_POINTER(ps_list)[i], NULL, S_evaluator);
			ps_element->name = (bIsNULL?EMPTY_STRING:(char*)make_name(CHARACTER_POINTER(ps_names)[i],S_evaluator)->text);
			LIST_POINTER(ps_list)[i] = ps_element;
		}

		ReAttachAndAssign(ps_list);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return bSuccess;
}

CSPcharacter CSPlist::GetNames(BOOL bValidate) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	return CSPcharacter(::get_names(&(*this)));
}

s_object*& CSPlist::GetAtDirect( const char* pszName, BOOL bValidate )
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	if ( !pszName || !*pszName )
		SCONNECT_ThrowException("Invalid name");

	s_object *ps_names = ::get_names(&(*this));

	if( SPL_NotThere(ps_names) )
		SCONNECT_ThrowException("List has no names");

	if ( !IS_CHARACTER(ps_names) )
		SCONNECT_ThrowException("List names are not character");
	long numNames = GET_LENGTH(ps_names);

	for ( long i=0; i<numNames; i++ )
	{
		if ( strcmp( CHARACTER_POINTER(ps_names)[i], pszName ) == 0 )
			return GetAtDirect(i, FALSE);
	}
	SCONNECT_ThrowException("Invalid name %s", pszName);
	return blt_in_NULL;
}

const s_object* CSPlist::GetAt( const char* pszName, BOOL bValidate ) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	const s_object* ps_result = blt_in_NULL;
	try
	{
		if ( !pszName || !*pszName )
			SCONNECT_ThrowException("Invalid name");
		s_object* ps_names = GetNames(FALSE);
		if( SPL_NotThere(ps_names) )
			return blt_in_NULL;

		if ( !IS_CHARACTER(ps_names) )
			SCONNECT_ThrowException("List names are not character");
		long numNames = GET_LENGTH(ps_names);

		for ( long i=0; i<numNames; i++ )
		{
			if ( strcmp( CHARACTER_POINTER(ps_names)[i], pszName ) == 0 )
			{
				ps_result = GetAt(i, FALSE);
				break;
			}
		}
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return ps_result;
}

s_object*& CSPlist::GetAt( const char* pszName, BOOL bValidate )
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	try
	{
		if ( !pszName || !*pszName )
			SCONNECT_ThrowException("Invalid name");
		s_object* ps_names = GetNames(FALSE);
		if( SPL_NotThere(ps_names) )
			return blt_in_NULL;

		if ( !IS_CHARACTER(ps_names) )
			SCONNECT_ThrowException("List names are not character");
		long numNames = GET_LENGTH(ps_names);

		for ( long i=0; i<numNames; i++ )
		{
			if ( strcmp( CHARACTER_POINTER(ps_names)[i], pszName ) == 0 )
			{
				return GetAt(i, FALSE);
				break;
			}
		}
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return blt_in_NULL;
}


// Remove element at index: zero-base
BOOL CSPlist::RemoveAt(long lElement, long nElements, BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		Attach(alcvec(S_MODE_LIST,(long)0,S_evaluator));

	BOOL bSuccess = FALSE;
	try
	{
		long length = GetLength(FALSE);

		if ( lElement < 0 || lElement >= length )
			SCONNECT_ThrowException("Invalid element specification", lElement); 
		if ( nElements <= 0 )
			SCONNECT_ThrowException("The number of elements to remove must be positive", nElements); 

		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator s;

		s_object* ps_list = CopyForWrite();

		for ( long i=lElement+nElements; i<length; i++ )
		{
			LIST_POINTER(ps_list)[i-nElements] = LIST_POINTER(ps_list)[i]; 
		}
		SET_LENGTH(ps_list, length-nElements);

		//Re-attach and re-assign if needed
		ReAttachAndAssign(ps_list);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return bSuccess;
}

// Remove element with name
BOOL CSPlist::RemoveAt(const char* pszName, BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		Attach(alcvec(S_MODE_LIST,(long)0,S_evaluator));

	BOOL bSuccess = FALSE;
	try
	{
		if ( !pszName || !*pszName )
			SCONNECT_ThrowException("Invalid name", CSPexception::Error);

		s_object* ps_names = GetNames(FALSE);
		if( SPL_NotThere(ps_names) || GET_LENGTH(ps_names)==0 )
			SCONNECT_ThrowException("Missing names slot");

		if ( !IS_CHARACTER(ps_names) )
			SCONNECT_ThrowException("Invalid object names");

		long numNames = GET_LENGTH(ps_names);
		long numElements = GetLength(FALSE);
		if ( numNames != numElements )
			SCONNECT_ThrowException("Invalid names vector");

		for ( long i=0; i<numNames; i++ )
		{
			char *pszNameFound = CHARACTER_POINTER(ps_names)[i];
			if ( strcmp( pszNameFound, pszName ) == 0 )
			{
				bSuccess = RemoveAt(i, 1, FALSE);
				break;
			}
		}
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return bSuccess;
}

// Return pointer to the data part which is a list object.
// The list object contains a list of objects.
s_object* CSPlist::GetData(BOOL bValidate) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	return GetPtr(); //return the list object
}

CSPlist::CProxy CSPlist::operator[](const char* pszName)
{
	if(!IsValid())
		SCONNECT_ThrowException("Invalid object");
	if ( !pszName || !*pszName )
		SCONNECT_ThrowException("Invalid name");
	s_object* ps_names = GetNames(FALSE);
	if( SPL_NotThere(ps_names) )
		SCONNECT_ThrowException("List has no names");

	if ( !IS_CHARACTER(ps_names) )
		SCONNECT_ThrowException("List names are not character");
	long numNames = GET_LENGTH(ps_names);

	for ( long i=0; i<numNames; i++ )
	{
		if ( strcmp( CHARACTER_POINTER(ps_names)[i], pszName ) == 0 )
			return CSPlist::CProxy(*this, i);
	}
	SCONNECT_ThrowException("Invalid name %s", pszName);
	return CSPlist::CProxy(*this, 0);
}

CSPlist::CProxy CSPlist::operator() (const char* pszName)
{
	if(!IsValid())
		SCONNECT_ThrowException("Invalid object");
	if ( !pszName || !*pszName )
		SCONNECT_ThrowException("Invalid name");
	s_object* ps_names = GetNames(FALSE);
	if( SPL_NotThere(ps_names) )
		SCONNECT_ThrowException("List has no names");

	if ( !IS_CHARACTER(ps_names) )
		SCONNECT_ThrowException("List names are not character");
	long numNames = GET_LENGTH(ps_names);

	for ( long i=0; i<numNames; i++ )
	{
		if ( strcmp( CHARACTER_POINTER(ps_names)[i], pszName ) == 0 )
			return CSPlist::CProxy(*this, i);
	}
	SCONNECT_ThrowException("Invalid name %s", pszName);
	return CSPlist::CProxy(*this, 0);
}

//////////////////////////////////////////////////////
// Proxy implementation
CSPlist::CProxy::CProxy(CSPlist& sVector, long lZeroBasedIndex)
:TSPvector<s_object*, S_MODE_LIST>::CSPvectorProxy(sVector, lZeroBasedIndex)
{
}		
CSPlist::CProxy::CProxy(const CProxy& proxyRhs)
:TSPvector<s_object*, S_MODE_LIST>::CSPvectorProxy(proxyRhs.m_vector, proxyRhs.m_lIndex)
{
}		

CSPlist::CProxy& CSPlist::CProxy::operator=(long lRhs)
{
	s_object* ps_object = NEW_INTEGER(1);
	INTEGER_POINTER(ps_object)[0]= lRhs;
	static_cast<CSPlist&>(m_vector).SetAt(m_lIndex, ps_object, FALSE);
	return *this;
}

CSPlist::CProxy& CSPlist::CProxy::operator=(double dRhs)
{
	s_object* ps_object = NEW_NUMERIC(1);
	NUMERIC_POINTER(ps_object)[0]= dRhs;
	static_cast<CSPlist&>(m_vector).SetAt(m_lIndex, ps_object, FALSE);
	return *this;
}

CSPlist::CProxy& CSPlist::CProxy::operator=(const s_object* ps_object)
{
	static_cast<CSPlist&>(m_vector).SetAt(m_lIndex, const_cast<s_object*>(ps_object), FALSE);
	return *this;
}

CSPlist::CProxy& CSPlist::CProxy::operator=(const CSPlist::CProxy& proxyRhs)
{
	s_object* ps_object = static_cast<CSPlist&>(proxyRhs.m_vector).GetAtDirect(proxyRhs.m_lIndex);
	static_cast<CSPlist&>(m_vector).SetAt(m_lIndex, ps_object, FALSE);
	return *this;
}

