package Splus::SHLIB;
use v5.8;
use strict ;

=head1 NAME

  Splus::SHLIB - make shared library (dll) from C or Fortran source

=head1 SYNOPSIS

  The following is the plan.  Only the ones before the bar are implemented.

  use Splus::SHLIB;
  $sh = Splus::SHLIB->new(@ARGV) ;
  $sh->make() ; # make the library
  $sh->get_make_command() ; # the make command used by make().
  $sh->add_source_files("foo.c");
  $sh->delete_source_files("foo.c");
  --- bar ---
  $sh->add_cflag("-DDEBUG");
  $sh->delete_cflag("-DDEBUG");
  $sh->append_argument("name=value");
  $sh->tidy_after_build() ; # remove object files, but not DLL
  $sh->make_virgin() ; # remove all generated files
  $sh->make_from_scratch(); # make_virgin then make

=cut

use Splus::Vars;
use Splus::Make;
use Splus::SplusUtils ;
use Cwd ;

Splus::Vars::error("SHOME");
my $SHOME=${Splus::Vars::SHOME} ;
my $ms = ${Splus::Vars::S_BUILD_SYS} =~ /^MS/i ;
if ($ms && $SHOME =~ / /) { $SHOME=dosify($SHOME); } 

Splus::Vars::error("MAKE");
Splus::Vars::error("OSTYPE");

my $osname = ${Splus::Vars::OSTYPE} eq "windows" ? "windows" : $^O ; # "windows", "linux", "solaris", "aix", 

my $obj_suffix = $ms ? '.obj' : '.o' ;

my $shlib_suffix = $ms ? '.dll' : '.so' ;

sub _Syntax {
    print "SHLIB: compile and link C and Fortran code files to make shared/dynamic library\n";
    print "-h,--help          # print this help file\n";
    print "-o libName         # name of library to make (don't include .so or .dll)\n";
    print "--clean-first      # remove all installed Splus objects reinstalling functions\n";
    print "--clean-after      # remove intermediate files not needed to use library\n";
    print "All .c and .f files in current directory will be compiled and linked\n";
    print "You can use file1.c file2.c to install just a subset of them\n";
    @_ && die join(", ", @_) ;
}


sub find_makevars_file {
    my $dir = shift ;
    $dir = dosify($dir) ;
    my $template="Makevars_" . "$Splus::Vars::OSTYPE" ;
    $template=("$template" . "_" . "$Splus::Vars::S_BUILD_SYS") if "$Splus::Vars::S_BUILD_SYS" ne "" ;
    my $filename="";
    LAB: until ( $template eq ""  ) {
        $filename = dosify("$dir/$template");
        if ( -e $filename) { last LAB ; }
        $template =~ s/_?[^_]*$// ;
    }
    $filename=undef if ($template eq "") ;
    $filename ;
}

sub new {
    # SHLIB -o shlib file1.c file2.c file3.f
    # Will make shlib.{dll,so,sl}, according to OS, from the C or Fortran files
    my $class = shift ;
    my $self = {} ;
    $self->{uses_fortran} = 0;
    $self->{uses_c} = 0;
    $self->{uses_cplusplus} = 0;
    $self->{objs} = ();
    $self->{srcs} = {};
    $self->{pkg_libs} = ();
    $self->{shlib} = undef ;
    $self->{make_cmd} = dosify(${Splus::Vars::MAKE}) ;
    my $n_src_arg = 0 ;

    my @args = globlist(@_) ; # *.c -> a.c b.c ...

    $self->{clean_after} = 0 ;
    while (@args) {
        my $arg = shift @args ;
        if ($arg =~ /\.(f|c|cc|cpp|cxx)$/i) {
            ${$self->{srcs}}{"$arg"}=undef ;
            $n_src_arg++ ;
        } elsif ($arg =~ /^-[lL]/ ) {
            push @{$self->{pkg_libs}}, $arg ;
        } elsif ($arg =~ /^-o/) {
            $self->{shlib} = shift @args ;
        } elsif ($arg =~ /^(-v|--verbose)/) {
            $self->{verbose} = 1 ;
        } elsif ($arg =~ /^(-h|--help)/) {
            _Syntax();
            exit();
        } elsif ($arg =~ /^--clean-first/) {
            $self->{clean_first} = 1 ;
        } elsif ($arg =~ /^--clean-after/) {
            $self->{clean_after} = 1 ;
        } elsif ($arg =~ /^--no-clean-after/) {
            $self->{clean_after} = 0 ;
        } else {
            warn "argument \"$arg\" is not recognized, ignored\n";
        }
    }

    $self->{shlib} ne "" or _Syntax "No -o libraryname argument found" ;
    
    if (! $n_src_arg ) {
       @args = globlist(qw(*.c *.C *.f *.F *.cc *.cpp *.cxx *.CC *.CPP *.CXX)) ;
       while (@args) {
            my $arg = shift @args ;
            ${$self->{srcs}}{"$arg"}=undef ;
            
       }
    }

    $self->{shlib} =~ s/\.(so|dll|DLL)// ; # remove possible suffix from output library name
    $self->{shlib} = dosify($self->{shlib});

    # now find the Makevars files in the standard directories
    # and get macro definitions from them.
    my $mk = Splus::Make->new() ;
    my $makevars_dir ;
    foreach $makevars_dir (".", "$SHOME/cmd") {
       my $makevars_file=&find_makevars_file($makevars_dir);
       if ($makevars_file) {
          $mk->parse_makefile($makevars_file) ;
          push @{$self->{makevars_files}}, $makevars_file;
       }
    }
    $self->{makevars} = $mk ;

    # And find what PKG_LIBS they defined, because we may need to process it.
    my $PKG_LIBS=$mk->expand_makevar("PKG_LIBS") ;
    my @pkg_libs=split(/\s+/, $PKG_LIBS);
        
    if ($ms) {
        my $pkg_lib ;
        foreach $pkg_lib (@pkg_libs) {
           $pkg_lib =~ s/^-l(.*)/$1.lib/ ;
           if ($pkg_lib =~ /^-L/) {
             my $pkg_libdir = $' ; # the part of the string after the matched ^-L
             $pkg_libdir = dosify($pkg_libdir) ;
             $pkg_lib =~ s/^-L(.*)/\/LIBPATH:$pkg_libdir/ ;
             push @{$self->{pkg_libs}}, $pkg_lib ;
           }
        }
    }
    _process_srcs($self) ;
    
    bless $self, $class ;
}

sub _process_srcs
{
    # not for end-user use
    # No arguments (except implicit $self).
    # Take set of src files (a hash whose keys are the file names)
    # and generate objs list and the flags uses_fortran and uses_c.
    my $self = shift ;
    $self->{uses_fortran} = 0;
    $self->{uses_c} = 0;
    $self->{objs} = [] ;
    foreach my $src (keys(%{$self->{srcs}})) {
        if ($src =~ /\.(f|c|cc|cpp|cxx)$/i) {
            $src = dosify($src) ;
            my $obj = $src ; $obj =~ s/\.(f|c|cc|cpp|cxx)$/$obj_suffix/i ;
            $self->{uses_fortran} |= $src=~/\.[f]$/i ;
            $self->{uses_c} |= $src=~/\.[c]$/i ;
            $self->{uses_cplusplus} |= $src=~/\.(cc|cpp|cxx)$/i ;
            push @{$self->{objs}}, $obj ;
        }
    }
}

sub add_source_files
{
    my $self = shift ;
    foreach my $source_file (globlist(@_)) {
        ${$self->{srcs}}{"$source_file"}=undef ; # ignore duplicates (could use ++ and warn if >1)
    }
    $self->_process_srcs() ;
}
sub delete_source_files
{
    my $self = shift ;
    foreach my $source_file (globlist(@_)) {
        delete ${$self->{srcs}}{"$source_file"} ;
    }
    $self->_process_srcs() ;
}

sub get_source_files
{
    my $self = shift ;
    keys %{$self->{srcs}} ;
}
sub get_obj_files
{
    my $self = shift ;
    @{$self->{objs}} ;
}
sub make_clean
{
    # remove object files and, if feasible, other
    # things made during build that are not needed
    # to use the shared library.
    my $self = shift ;
    print "SHLIB: make_clean ...\n";
    foreach my $obj (@{$self->{objs}}) {
        if ( -e $obj) {
            unlink $obj or warn "Could not remove object file \"$obj\" (error: $!)" ;
        }
    }
    # On Windows there are a bunch of files whose name is based on shlib name (.def,.idb)
    foreach my $file ("$self->{shlib}.idb", "$self->{shlib}.def") {
        if ( -e $file) {
            unlink $file or warn "Could not remove object file \"$file\" (error: $!)" ;
        }
    }
}

sub make_virgin
{
    my $self = shift ;
    print "SHLIB: make_virgin ...\n";
    $self->make_clean() ;
    unlink($self->_shlib_full_name()) ;
    -e $self->_shlib_full_name() and warn "Cannot remove " . abs_path($self->_shlib_full_name()) . "\n" ;
}

sub get_make_command
{
    my $self = shift ;
    # For now, ignore those processed PKG_LIBS.  They may have $'s and ('s
    # in them and require quoting to pass tosystem, but quoting depends on
    # the shell (/bin/sh vs. cmd.exe).
    my $makevars_args = "" ;
    my $makevars_file ;
    foreach $makevars_file (@{$self->{makevars_files}}) {
       if ($makevars_file) {
          $makevars_args .=" -f " . dosify("$makevars_file") ;
       }
    }
    my $command="$self->{make_cmd} SHOME=$SHOME $makevars_args SHLIB_NAME=$self->{shlib} OBJS=\"" . join(" ", @{$self->{objs}}) . "\" " ;
    
    $command ;
}

sub make
{
    my $self = shift ;
    if ($#{$self->{objs}} >= 0) {
        $self->{clean_first} and $self->make_virgin() ;
        my $command = $self->get_make_command() ;
        my $status = system $command ;
        $self->{clean_after} and $self->make_clean() ;
        !$status or die "Problem running the make command";
    } else {
        print "No source code to make into dynamic/shared library\n" if $self->{verbose}  ;
    }
}

sub _shlib_full_name
{
    my $self = shift ;
    $self->{shlib} . ".$shlib_suffix" ;
}

sub check_for_unresolved_symbols
{
    my $self = shift ;
    $self->{verbose} and print "$0: checking for unresolved symbols in shared/dynamic library\n";
    my @undefined_syms ;
    SWITCH: for ($osname) {
               /windows/     && do { @undefined_syms = self->check_for_unresolved_symbols_windows(@_) ; last; };
               /linux/       && do { @undefined_syms = self->check_for_unresolved_symbols_linux(@_) ; last; };
               /solaris/     && do { @undefined_syms = self->check_for_unresolved_symbols_solaris(@_) ; last; };
               warn "check_for_unresolved_symbols is not implemented on the $osname operating system" ;
           }
    @undefined_syms ;
}

sub check_for_unresolved_symbols_windows
{
    my $self = shift ;
    my @undefined_syms ;
    warn "check_for_unresolved_symbols is not implemented on the $osname operating system" ;
    my $shlib = $self->_shlib_full_name() ;
    $self->make() if ! -e $shlib ;
    if (! -e $shlib) {
        warn "No shared library made, so cannot check for unresolved symbols";
        return @undefined_syms ;
    }
    @undefined_syms ;
}

sub check_for_unresolved_symbols_solaris
{
    my $self = shift ;
    my @undefined_syms ;
    warn "check_for_unresolved_symbols is not completely implemented on the $osname operating system" ;
    my $shlib = $self->_shlib_full_name() ;
    $self->make() if ! -e $shlib ;
    if (! -e $shlib) {
        warn "No shared library made, so cannot check for unresolved symbols";
        return @undefined_syms ;
    }
    my $cmd = "elfdump -s -N .dynsym $shlib" ;
    open my $nmpipe, "$cmd|" or die "Cannot run $cmd ($!)" ;
    while (<$nmpipe>) {
        chomp ;
        # Looks for entries like:
        #       [10]  0x00000000 0x00000000  NOTY GLOB  D    0 UNDEF       no_such_datum
        #       [12]  0x00000000 0x00000000  FUNC GLOB  D    0 UNDEF       no_such_function
        # This will still get false positives - things in dependent libraries have same pattern
        if (/^\s*\[\d+\]\w+0x0+\s+0x0+\w+(NOTY|FUNC)\s+GLOB\s+D\s+0\s+UNDEF\s+/) {
            push @undefined_syms, $';
        }
    }
    @undefined_syms ;
}

sub check_for_unresolved_symbols_linux
{
    my $self = shift ;
    my @undefined_syms ;
    my $shlib = $self->_shlib_full_name() ;
    $self->make() if ! -e $shlib ;
    if (! -e $shlib) {
        warn "No shared library made, so cannot check for unresolved symbols";
        return @undefined_syms ;
    }
    my $cmd = "objdump --dynamic-syms $shlib" ;
    open my $nmpipe, "$cmd|" or die "Cannot run $cmd ($!)" ;
    while (<$nmpipe>) {
        chomp ;
        if (/^0+\s+D\s+\*UND\*\s+0+\s+/) {
            push @undefined_syms, $';
        }
    }
    close $nmpipe ;
    print "undefined_syms=" . join(", ", @undefined_syms), "\n";
    @undefined_syms ;
}

1;
