/* -*- mode: C; c-file-style: "gnu"; indent-tabs-mode: nil; -*-
 *
 * Copyright © 2018 Endless Mobile, Inc.
 *
 * SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Authors:
 *  - Philip Withnall <withnall@endlessm.com>
 */

#include <gio/gio.h>
#include <glib.h>
#include <libgsystemservice/peer-manager.h>
#include <libgsystemservice/peer-manager-dbus.h>
#include <locale.h>
#include <unistd.h>


typedef struct
{
  GTestDBus *bus;  /* (owned) */
} Fixture;

static void
setup (Fixture       *fixture,
       gconstpointer  user_data)
{
  fixture->bus = g_test_dbus_new (G_TEST_DBUS_NONE);
  g_test_dbus_up (fixture->bus);
}

static void
teardown (Fixture       *fixture,
          gconstpointer  user_data)
{
  if (fixture->bus != NULL)
    g_test_dbus_down (fixture->bus);
  g_clear_object (&fixture->bus);
}

static void
async_result_cb (GObject      *object,
                 GAsyncResult *result,
                 gpointer      user_data)
{
  GAsyncResult **result_out = user_data;

  g_assert (*result_out == NULL);
  *result_out = g_object_ref (result);

  g_main_context_wakeup (g_main_context_get_thread_default ());
}

static void
test_construction (Fixture       *fixture,
                   gconstpointer  user_data)
{
  g_autoptr(GError) local_error = NULL;
  g_autoptr(GDBusConnection) connection = NULL;
  g_autoptr(GssPeerManager) manager = NULL;

  g_test_summary ("Test that no credentials are returned for a nonexistent peer.");

  connection = g_bus_get_sync (G_BUS_TYPE_SESSION, NULL, &local_error);
  g_assert_no_error (local_error);

  manager = GSS_PEER_MANAGER (gss_peer_manager_dbus_new (connection));

  g_assert_null (gss_peer_manager_get_peer_argv0 (manager, ":999.999"));
  g_assert_cmpint (gss_peer_manager_get_peer_pidfd (manager, ":999.999"), <, 0);
  g_assert_cmpint (gss_peer_manager_get_peer_uid (manager, ":999.999"), ==, (uid_t) -1);
}

static void
test_self_peer (Fixture       *fixture,
                gconstpointer  user_data)
{
  g_autoptr(GError) local_error = NULL;
  g_autoptr(GDBusConnection) connection = NULL;
  const char *connection_unique_name;
  g_autoptr(GssPeerManager) manager = NULL;
  g_autoptr(GAsyncResult) result = NULL;
  gboolean retval;

  g_test_summary ("Test that correct credentials are returned for this process.");

  connection = g_bus_get_sync (G_BUS_TYPE_SESSION, NULL, &local_error);
  g_assert_no_error (local_error);
  connection_unique_name = g_dbus_connection_get_unique_name (connection);

  manager = GSS_PEER_MANAGER (gss_peer_manager_dbus_new (connection));

  gss_peer_manager_ensure_peer_credentials_async (manager,
                                                  connection_unique_name,
                                                  NULL,
                                                  async_result_cb,
                                                  &result);
  while (result == NULL)
    g_main_context_iteration (NULL, TRUE);
  retval = gss_peer_manager_ensure_peer_credentials_finish (manager, result, &local_error);
  g_assert_no_error (local_error);
  g_assert_true (retval);

  g_assert_nonnull (gss_peer_manager_get_peer_argv0 (manager, connection_unique_name));
  g_assert_true (g_str_has_suffix (gss_peer_manager_get_peer_argv0 (manager, connection_unique_name), "/peer-manager"));
  g_assert_cmpint (gss_peer_manager_get_peer_pidfd (manager, connection_unique_name), >=, 0);
  g_assert_cmpint (gss_peer_manager_get_peer_uid (manager, connection_unique_name), ==, getuid ());
}

int
main (int   argc,
      char *argv[])
{
  setlocale (LC_ALL, "");

  g_test_init (&argc, &argv, NULL);

  g_test_add ("/peer-manager/construction", Fixture, NULL, setup,
              test_construction, teardown);
  g_test_add ("/peer-manager/self-peer", Fixture, NULL, setup,
              test_self_peer, teardown);

  return g_test_run ();
}
