% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/doc-selectors.R
\name{flextable_selectors}
\alias{flextable_selectors}
\title{Selectors in flextable}
\description{
Selectors are a core feature of flextable that allow you to specify which
parts (\code{part}), rows (\code{i}) and columns (\code{j}) should be affected by formatting,
styling, or content operations.

Many flextable functions support these selectors, including \code{\link[=bg]{bg()}},
\code{\link[=bold]{bold()}}, \code{\link[=color]{color()}}, \code{\link[=padding]{padding()}}, \code{\link[=fontsize]{fontsize()}}, \code{\link[=italic]{italic()}}, \code{\link[=align]{align()}},
\code{\link[=mk_par]{mk_par()}}, \code{\link[=hline]{hline()}}, \code{\link[=vline]{vline()}}, and many others.

Selectors make conditional formatting easy and enable seamless piping of
multiple operations using \verb{\%>\%} or \verb{|>}.
}
\section{Row selection with the \code{i} parameter}{


The \code{i} parameter is used to select specific rows in a flextable for
formatting, styling, or content operations.

When \code{i = NULL} (the default), operations apply to all rows.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{i = ~ condition           # Formula (body only)
i = 1:5                   # Integer vector
i = c(TRUE, FALSE, ...)   # Logical vector
i = NULL                  # All rows (default)
}\if{html}{\out{</div>}}

Best practices:
\itemize{
\item Use formulas (\code{i = ~ condition}) for conditional selection in body
\item Use integers for positional selection in any part.
\item Use \code{nrow_part()} for dynamic row selection
}
}

\section{Column selection with the \code{j} parameter}{

The \code{j} parameter is used to select specific columns in a flextable for
formatting, styling, or content operations.

When \code{j = NULL} (the default), operations apply to all columns.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{j = ~ col1 + col2         # Formula (select multiple)
j = ~ . - col1            # Formula (exclude columns)
j = c("col1", "col2")     # Character vector (recommended)
j = 1:5                   # Integer vector
j = c(TRUE, FALSE, ...)   # Logical vector
j = NULL                  # All columns (default)
}\if{html}{\out{</div>}}

Best practices:
\itemize{
\item Use character vectors (\code{j = c("col1", "col2")}) for clarity and
maintainability.
\item Use formulas (\code{j = ~ col1 + col2}) for excluding columns.
\item Avoid integer positions when possible (less maintainable).
\item Column selectors work with all parts (header, body, footer, all).
}
}

\section{Part selection with the \code{part} parameter}{


The \code{part} parameter specifies which section of the flextable should be
affected by formatting, styling, or content operations.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{part = "body"    # Data rows (default)
part = "header"  # Header rows
part = "footer"  # Footer rows
part = "all"     # All parts
}\if{html}{\out{</div>}}
\itemize{
\item When \code{part = "body"} (the default), operations apply only to the data rows.
\item When \code{part = "all"}, the operation is applied to each part independently.
}

Formula row selectors (\code{i = ~ condition}) CANNOT be used with:
\itemize{
\item \code{part = "header"} - Headers contain only character values
\item \code{part = "footer"} - Footers contain only character values
\item \code{part = "all"} - Header/footer are character-only while body has original data types
}

This restriction exists because formula selectors evaluate conditions using
the actual data types from your dataset (numeric, logical, etc.). Headers and
footers store only character representations of values, so conditional
expressions like \code{i = ~ price < 330} cannot be evaluated on them.
}

\section{Advanced Programming with Selectors}{


Function \code{\link[=nrow_part]{nrow_part()}} returns the number of lines in a part.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Format the last row differently
ft <- flextable(iris[48:52,])
ft <- bold(ft, i = nrow_part(ft, part = "body"))

# Add footnote to last row
ft <- footnote(
  ft,
  i = nrow_part(ft, part = "body"), j = 1:4,
  value = as_paragraph("Calculated mean")
)
ft
}\if{html}{\out{</div>}}

Function \code{\link[=ncol_keys]{ncol_keys()}} returns the number of columns.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Get column count
ncol_keys(ft)

# Useful for programmatic selection
ncols <- ncol_keys(ft)
ft <- bg(ft, j = rep(c(TRUE, FALSE), length.out = ncols), bg = "#f0f0f0")
ft
}\if{html}{\out{</div>}}
}

\concept{selectors in flextable}
\keyword{internal}
